#ifndef __CDriveInformation__
#define __CDriveInformation__

#include "../Basics/CString.hpp"
#include "../Basics/CSystemString.hpp"
#include "../Collections/CollectionTypedef.h"
#include "../IO/CTextStream.hpp"

//	===========================================================================

using Exponent::Basics::CString;
using Exponent::Basics::CSystemString;
using Exponent::IO::CTextStream;
using Exponent::Collections::TStringCountedPointerArray;

//	===========================================================================

namespace Exponent
{
	namespace Host
	{
		/**
		 * @class CDriveInformation CDriveInformation.hpp
		 * @brief Gives informaiton about the disk structure on the host system
		 *
		 * @date 26/09/2004
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 * @version 1.1.0 Made GetDriveNames take an inplace TStringArray
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: CDriveInformation.hpp,v 1.4 2007/02/08 21:06:44 paul Exp $
		 */
		class CDriveInformation
		{
		public:

//	===========================================================================

/** @cond */
#ifdef WIN32
/** @endcond */
			static const long CDRIVE_INFO_UNKNOWN_DRIVE		= DRIVE_UNKNOWN;			/**< Unknown unit */
			static const long CDRIVE_INFO_UNMOUNTED_DRIVE	= DRIVE_NO_ROOT_DIR;		/**< Unmounted drive */
			static const long CDRIVE_INFO_REMOVEABLE_DRIVE	= DRIVE_REMOVABLE;			/**< Drive is removeable */
			static const long CDRIVE_INFO_FIXED_DRIVE		= DRIVE_FIXED;				/**< Drive is fixed */
			static const long CDRIVE_INFO_NETWORK_DRIVE		= DRIVE_REMOTE;				/**< Drive is remote (network) */
			static const long CDRIVE_INFO_CDROM_DRIVE		= DRIVE_CDROM;				/**< Drive is a CDROM */
			static const long CDRIVE_INFO_RAM_DRIVE			= DRIVE_RAMDISK;			/**< Drive is a ram disk */
/** @cond */
#endif
/** @endcond */

//	===========================================================================

			/**
			 * Dump the info to a text file
			 * @param path The path to write the data to
			 */
			static void dumpDriveInfoToFile(const CSystemString &path);

			/**
			 * Dump the info to a stream output
			 * @param stream The stream to use
			 */
			static void dumpDriveInfoToFile(CTextStream &stream);

//	===========================================================================

			/**
			 * Get the name of all valid drives
			 * @param array The array to fill
			 */
			static void getDriveNames(TStringCountedPointerArray &array);

			/**
			 * Get drive information - drive to examine should be of form "c:\" or "network name\network drive name\"\n
			 * NOTES : On mac the volumeSerialNumber is actually the volume number (starting at 0 i believe)
			 * @param driveToExamine The drive to examine
			 * @param volumeName On return is filled with the volume nmae
			 * @param fileSystem On return is filled with a string identifying the filesystem
			 * @param volumeSerialNumber On return is filled with the volume serial number
			 */
			static void getDriveInformation(const CString &driveToExamine, CString &volumeName, CString &fileSystem, long &volumeSerialNumber);

//	===========================================================================

/** @cond */
#ifdef WIN32
/** @endcond */
			/**
			 * Get the drive type
			 * @param driveToExamine The drive to examine
			 * @retval long The type of drive
			 */
			static long getDriveType(const CString &driveToExamine);

			/**
			 * Is the drive a cdrom
			 * @param driveToExamine The drive to examine
			 * @retval bool True if drive is CDROM, false otherwise
			 */
			static bool driveIsCdrom(const CString &driveToExamine);

			/**
			 * Is the drive a network drive
			 * @param driveToExamine The drive to examine
			 * @retval bool True if drive is network drive, false otherwise
			 */
			static bool driveIsNetworkDrive(const CString &driveToExamine);

			/**
			 * Is local hard disk
			 * @param driveToExamine The drive to examine
			 * @retval bool True if drive is local hard disk, false otherwise
			 */
			static bool driveIsFixedDrive(const CString &driveToExamine);

			/**
			 * Get the drive type as a string
			 * @param driveToExamine The drive to examine
			 * @param type On return stores the type of drive as a string
			 */
			static void getDriveTypeAsString(const CString &driveToExamine, CString type);

//	===========================================================================

			/**
			 * Get the windows temp folder
			 * @param folder On return stores the path to the windows temp folder
			 */
			static void getWindowsTemporaryFolder(CSystemString &folder);

			/**
			 * Get the windows folder
			 * @param folder On return stores the path to the windows folder
			 */
			static void getWindowsFolder(CSystemString &folder);
/** @cond */
#endif
/** @endcond */

//	===========================================================================

		private:

//	===========================================================================

/** @cond */
#ifdef WIN32
/** @endcond */
			static const char *CDRIVE_INFO_DRIVE_STRINGS[];		/**< Drive types as strings */
/** @cond */
#endif
/** @endcond */

//	===========================================================================
		};
	}
}
#endif	// End of CDriveInformation.hpp